// **proxy.js**：前端链接拦截器 (Production Optimized)
const JS_PROXY_REWRITER = `
(function() {
    // 使用 Base64 编码 (与后端 PHP urlsafe_base64_encode 对应)
    function encodeUrl(url) {
        return btoa(url).replace(/\\+/g, '-').replace(/\\//g, '_').replace(/=+$/, '');
    }

    function rewriteUrl(originalUrl) {
        if (!originalUrl) return originalUrl;
        
        // 1. 已经是代理链接则跳过
        if (originalUrl.includes(location.origin + '/index.php') || originalUrl.includes('?u=')) return originalUrl;
        
        // 2. 忽略特殊协议
        if (originalUrl.startsWith('#') || originalUrl.startsWith('javascript:') || originalUrl.startsWith('data:')) return originalUrl;

        try {
            const absolute = new URL(originalUrl, location.href);
            // 3. 只有跨域请求才代理
            if (absolute.origin === location.origin) return originalUrl;

            // 4. 生成代理链接：使用 index.php?u=Base64 格式
            // 确保对 URL 进行 URI 编码以支持中文路径
            const encodedAbsolute = encodeUrl(absolute.href); 
            return location.origin + '/index.php?u=' + encodedAbsolute;
        } catch(e) {
            return originalUrl;
        }
    }

    function rewriteElement(el, attr) {
        const val = el.getAttribute(attr);
        if (val) {
            const newVal = rewriteUrl(val);
            if (newVal !== val) el.setAttribute(attr, newVal);
        }
    }

    function rewriteAll() {
        document.querySelectorAll('a[href]').forEach(el => rewriteElement(el, 'href'));
        document.querySelectorAll('img[src], script[src], iframe[src]').forEach(el => rewriteElement(el, 'src'));
        document.querySelectorAll('form[action]').forEach(el => rewriteElement(el, 'action'));
    }

    // 拦截 Fetch API
    const originalFetch = window.fetch;
    window.fetch = function(resource, config) {
        let url = resource;
        if (resource instanceof Request) url = resource.url;
        
        const newUrl = rewriteUrl(url);
        
        if (newUrl === url) {
             // 如果 URL 未更改，则直接调用原始 Fetch
             return originalFetch.apply(this, arguments);
        }
        
        // URL 已更改，构造新的请求
        if (resource instanceof Request) {
            // 需要构造一个新的 Request 对象，以新的 URL 替换旧的
            resource = new Request(newUrl, resource);
        } else if (typeof resource === 'string') {
            resource = newUrl;
        } else {
             // 无法处理的资源类型，回退
             return originalFetch.apply(this, arguments);
        }

        return originalFetch.call(this, resource, config);
    };

    // 监听 DOM 变化
    document.addEventListener('DOMContentLoaded', rewriteAll);
    const mo = new MutationObserver(rewriteAll);
    mo.observe(document.body, { childList: true, subtree: true, attributes: true });
})();
`;