<?php
/**
 * Advanced PHP Proxy / Accelerator v3.5 (中文路径修复 & Service Worker 融合版)
 * 核心修复：
 * 1. 修复 HTML/CSS 重写时，中文或非 ASCII 字符路径（如 /uncensored/巨乳）未进行 URL 编码导致链接失效的问题。
 * 2. 媒体文件默认内联显示，raw.githubusercontent.com 文本内容不重写。
 * 3. 融合 proxy.js (前端重写) 和 proxy-sw.js (Service Worker) 逻辑到此文件。
 */
declare(strict_types=1);

// 禁用输出缓冲，确保流式传输顺畅
if (ob_get_level()) ob_end_clean();

// ================= 配置区域 =================
// 增加脚本执行时间，防止大文件下载中途超时
set_time_limit(0); 
ini_set('memory_limit', '256M'); // 仅用于 HTML 处理时的临时缓冲

// 【重要】自定义加密密钥
define('ENCRYPT_KEY', 'G7k9#mE4!qW2aZ8xL5vR3tY6uP9nJ1hB@zX8cV5bN3mK2jH7fD4sA1');
define('MAX_FILE_SIZE', 0); // 0 = 无限制
define('UA_INDEX_FILE', 'ua_index.txt');

// 2025 最新 User-Agent 池 (模拟真实浏览器以绕过反爬)
const USER_AGENTS = [
    // 桌面浏览器
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Safari/537.36 Edg/131.0.0.0',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 15_0) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.0.0 Safari/537.36',
    'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:132.0) Gecko/20100101 Firefox/132.0',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 15.0; rv:132.0) Gecko/20100101 Firefox/132.0',
    // 移动端浏览器
    'Mozilla/5.0 (Linux; Android 15; Pixel 9) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.0.0 Mobile Safari/537.36',
    'Mozilla/5.0 (iPhone; CPU iPhone OS 18_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/18.1 Mobile/15G79 Safari/604.1',
    'Mozilla/5.0 (iPad; CPU OS 18_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/18.1 Mobile/15G79 Safari/604.1',
    'Mozilla/5.0 (Linux; Android 14; SM-G981B) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Mobile Safari/537.36',
    'Mozilla/5.0 (iPhone; CPU iPhone OS 18_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) CriOS/132.0.0.0 Mobile/15G72 Safari/604.1',
    'Mozilla/5.0 (Linux; Android 15; SM-S928B) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/132.0.0.0 Mobile Safari/537.36 EdgA/132.0.0.0',
];

// ========== 定义前端 JS 常量 (proxy.js 和 proxy-sw.js 的内容) ==========

// 1. Service Worker 核心逻辑 (proxy-sw.js)
const SW_CONTENT = 'const VERSION = "2.0.0";self.addEventListener("install", event => self.skipWaiting());self.addEventListener("activate", event => event.waitUntil(self.clients.claim()));self.addEventListener("fetch", event => {const req = event.request;const url = new URL(req.url);if (url.origin === location.origin || req.url.includes("/index.php?")) {return;}const targetUrl = btoa(req.url).replace(/\\+/g, "-").replace(/\\//g, "_").replace(/=+$/, "");const proxyUrl = `${location.origin}/index.php?u=${targetUrl}`;event.respondWith(fetch(proxyUrl, {headers: req.headers,mode: "cors",credentials: "omit"}).catch(() => {return fetch(req);}));});';

// 2. Service Worker 注册代码 (使用 Blob 动态创建)
$JS_SW_REGISTRATION = sprintf(
    '
(function() {
    if (\'serviceWorker\' in navigator) {
        window.addEventListener(\'load\', () => {
            const swBlob = new Blob([%s], { type: \'application/javascript\' });
            const swUrl = URL.createObjectURL(swBlob);
            
            navigator.serviceWorker.register(swUrl, { scope: \'/\' })
                .then(reg => console.log(\'SW registered:\', reg.scope))
                .catch(err => console.log(\'SW registration failed:\', err));
        });
    }
})();
',
    json_encode(SW_CONTENT)
);

// 3. 前端链接拦截器 (proxy.js)
const JS_PROXY_REWRITER = '
(function() {
    function encodeUrl(url) {
        return btoa(url).replace(/\\+/g, "-").replace(/\\//g, "_").replace(/=+$/, "");
    }

    function rewriteUrl(originalUrl) {
        if (!originalUrl) return originalUrl;
        if (originalUrl.includes(location.origin + "/index.php") || originalUrl.includes("?u=")) return originalUrl;
        if (originalUrl.startsWith("#") || originalUrl.startsWith("javascript:") || originalUrl.startsWith("data:")) return originalUrl;

        try {
            const absolute = new URL(originalUrl, location.href);
            if (absolute.origin === location.origin) return originalUrl;

            const encodedAbsolute = encodeUrl(absolute.href); 
            return location.origin + "/index.php?u=" + encodedAbsolute;
        } catch(e) {
            return originalUrl;
        }
    }

    function rewriteElement(el, attr) {
        const val = el.getAttribute(attr);
        if (val) {
            const newVal = rewriteUrl(val);
            if (newVal !== val) el.setAttribute(attr, newVal);
        }
    }

    function rewriteAll() {
        document.querySelectorAll("a[href]").forEach(el => rewriteElement(el, "href"));
        document.querySelectorAll("img[src], script[src], iframe[src], link[href][rel=\\"stylesheet\\"]").forEach(el => rewriteElement(el, "src"));
        document.querySelectorAll("form[action]").forEach(el => rewriteElement(el, "action"));
    }

    const originalFetch = window.fetch;
    window.fetch = function(resource, config) {
        let url = resource;
        if (resource instanceof Request) url = resource.url;
        
        const newUrl = rewriteUrl(url);
        
        if (newUrl === url) {
             return originalFetch.apply(this, arguments);
        }
        
        if (resource instanceof Request) {
            resource = new Request(newUrl, resource);
        } else if (typeof resource === "string") {
            resource = newUrl;
        } else {
             return originalFetch.apply(this, arguments);
        }

        return originalFetch.call(this, resource, config);
    };

    document.addEventListener("DOMContentLoaded", rewriteAll);
    const mo = new MutationObserver(rewriteAll);
    mo.observe(document.body, { childList: true, subtree: true, attributes: true });
})();
';


class ProxyHandler {
    private string $key;
    private string $JS_PROXY_REWRITER = JS_PROXY_REWRITER;
    private string $JS_SW_REGISTRATION;

    public function __construct() {
        $this->key = hash('sha256', ENCRYPT_KEY, true);
        global $JS_SW_REGISTRATION;
        $this->JS_SW_REGISTRATION = $JS_SW_REGISTRATION;
    }

    public function run(): void {
        $targetUrl = $this->getTargetUrl();

        if (!$targetUrl) {
            $this->renderForm();
            return;
        }

        if (!$this->securityCheck($targetUrl)) {
            http_response_code(403);
            die('Error: Access denied (SSRF Protection).');
        }

        $this->proxyRequest($targetUrl);
    }

    private function getTargetUrl(): ?string {
        // 1. 处理 POST 输入
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['target_url'])) {
            $url = trim($_POST['target_url']);
            $normalized = $this->normalizeUrl($url);
            if (filter_var($normalized, FILTER_VALIDATE_URL)) {
                $enc = urlencode($this->encryptAes($normalized));
                header("Location: ?url={$enc}");
                exit;
            }
        }
        
        // 2. 处理 ?url= 加密参数
        if (isset($_GET['url'])) {
            $decoded = $this->decryptAes($_GET['url']);
            return $this->validateAndNormalize($decoded);
        }

        // 3. 处理 ?u= Base64 参数
        if (isset($_GET['u'])) {
            $decoded = $this->urlsafe_base64_decode($_GET['u']);
            return $this->validateAndNormalize($decoded);
        }

        // 4. 兼容直接路径模式 /https://github.com...
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        $pathInfo = parse_url($uri, PHP_URL_PATH);
        $scriptName = $_SERVER['SCRIPT_NAME'];
        if (strpos($pathInfo, $scriptName) === 0) {
            $pathInfo = substr($pathInfo, strlen($scriptName));
        }
        
        // 检测 URL 模式
        if (preg_match('#^/(https?://.+)#i', $pathInfo, $matches)) {
            return $this->validateAndNormalize($matches[1]);
        }
        
        return null;
    }

    private function validateAndNormalize(string $url): ?string {
        $normalized = $this->normalizeUrl($url);
        return filter_var($normalized, FILTER_VALIDATE_URL) ? $normalized : null;
    }

    /**
     * 核心逻辑：智能流式代理
     */
    private function proxyRequest(string $url): void {
        $userAgent = USER_AGENTS[array_rand(USER_AGENTS)];

        // 第一步：初始化 cURL 仅获取 Header，判断文件类型
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_NOBODY => true, // 仅 HEAD
            CURLOPT_FOLLOWLOCATION => true, 
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => $userAgent,
            CURLOPT_HEADER => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
        ]);

        $headResponse = curl_exec($ch);
        
        // 获取最终跳转后的 URL
        $finalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL) ?: $url;
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE) ?: 'application/octet-stream';
        $contentLength = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        
        // 提取 Content-Disposition Header
        $upstreamDisposition = '';
        if ($headResponse && preg_match('/^Content-Disposition:\s*(.+)$/mi', $headResponse, $m)) {
            $upstreamDisposition = trim($m[1]);
        }

        curl_close($ch);

        // 如果 HEAD 失败，尝试直接 GET（容错）
        if (!$headResponse || $httpCode >= 400) {
            $this->streamDirectly($url, $userAgent);
            return;
        }

        // 判断是否为 raw.githubusercontent.com 
        $isRawGithub = (stripos(parse_url($finalUrl, PHP_URL_HOST) ?: '', 'raw.githubusercontent.com') !== false);

        // 判断类型
        $isHtml = stripos($contentType, 'text/html') !== false;
        $isText = stripos($contentType, 'text/') !== false;
        $isCss = stripos($contentType, 'text/css') !== false;

        // 特殊修正：通过扩展名辅助判断是否为二进制
        $path = parse_url($finalUrl, PHP_URL_PATH);
        if (preg_match('/\.(zip|rar|7z|tar|gz|exe|iso|mp4|mp3|pdf|apk)$/i', $path)) {
            $isHtml = false;
            $isCss = false;
            $isText = false; 
        }

        // 分支 1：如果是文本内容，使用缓冲模式进行处理 (决定是否重写)
        if ($isText) { 
            // 传递是否跳过重写的标志
            $this->processTextContent($finalUrl, $userAgent, $isCss, $isRawGithub);
        } 
        // 分支 2：如果是二进制文件（ZIP, 图片, 视频等）或未识别的类型，使用流式模式
        else {
            // 传递 $isRawGithub 标志用于控制 Content-Disposition
            $this->streamBinaryContent($finalUrl, $userAgent, $contentType, $contentLength, $upstreamDisposition, $isRawGithub);
        }
    }

    /**
     * 模式 A：流式传输二进制文件 (ZIP, 图片, 视频)
     */
    private function streamBinaryContent(string $url, string $ua, string $contentType, float $length, string $upstreamDisposition, bool $isRawGithub): void {
        header_remove(); 

        header("HTTP/1.1 200 OK");
        header("Content-Type: $contentType");
        if ($length > 0) {
            header("Content-Length: $length");
        }
        
        $isForcedDownload = true; 

        // 只有上游没有明确指定 Content-Disposition 时，才进行自定义判断
        if (empty($upstreamDisposition)) {
            if ($isRawGithub) {
                 $isForcedDownload = false;
            }
            
            // 规则 2: 如果内容类型是图片、视频或音频 (media)，则尝试内联显示。
            $typeGroup = strtok($contentType, '/');
            if (in_array($typeGroup, ['image', 'video', 'audio'], true)) {
                $isForcedDownload = false;
            }
        }

        // 文件名和 Content-Disposition 逻辑
        if (!empty($upstreamDisposition) && stripos($upstreamDisposition, 'filename') !== false) {
            header("Content-Disposition: $upstreamDisposition");
        } elseif ($isForcedDownload) {
            // 优先级 2: 强制下载模式 ( attachment )
            $filename = basename(parse_url($url, PHP_URL_PATH));
            $filename = rawurldecode($filename); 
            if (empty($filename) || $filename === '/') $filename = 'download_file.bin';
            header("Content-Disposition: attachment; filename=\"$filename\"");
        } else {
            // 优先级 3: 内联显示模式 ( inline )
            $filename = basename(parse_url($url, PHP_URL_PATH));
            $filename = rawurldecode($filename); 
            if (empty($filename) || $filename === '/') $filename = 'view.file';
            header("Content-Disposition: inline; filename=\"$filename\""); 
        }
        
        header("X-Proxy-Method: Stream-Mode" . ($isForcedDownload ? '-Attachment' : '-Inline'));

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => $ua,
            CURLOPT_BUFFERSIZE => 512 * 1024,
            CURLOPT_TIMEOUT => 3600,
            CURLOPT_WRITEFUNCTION => function($ch, $chunk) {
                echo $chunk;
                flush();
                return strlen($chunk);
            }
        ]);
        curl_exec($ch);
        curl_close($ch);
        exit;
    }

    /**
     * 模式 B：缓冲处理文本内容 (HTML, CSS, Text)
     */
    private function processTextContent(string $url, string $ua, bool $isCss, bool $skipRewriting = false): void {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => $ua,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '', // 自动解压 GZIP
            CURLOPT_TIMEOUT => 30
        ]);
        
        $content = curl_exec($ch);
        
        if ($content === false) {
            die('Error fetching resource: ' . curl_error($ch));
        }
        
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        curl_close($ch);

        // 再次检测 Magic Bytes 防止服务器 Content-Type 欺骗
        $realType = $this->detectContentTypeFromBytes($content);
        if ($realType && !str_starts_with($realType, 'text/')) {
             header("Content-Type: $realType");
             echo $content;
             return;
        }

        header("Content-Type: $contentType");
        
        // 根据 $skipRewriting 标志决定是否重写链接
        if ($skipRewriting) {
            header("X-Proxy-Method: Direct-Raw");
            echo $content;
        } else {
            header("X-Proxy-Method: Rewrite-Mode");
            if ($isCss) {
                echo $this->rewriteCss($content, $url);
            } else {
                echo $this->rewriteHtml($content, $url);
            }
        }
    }
    
    // === 核心修复函数 ===

    private function encodeUrlPath(string $url): string {
        // 使用 preg_replace_callback 查找非 ASCII 字符并进行 URL 编码
        return preg_replace_callback('/[^\x20-\x7E]+/u', function ($match) {
            return urlencode($match[0]);
        }, $url);
    }

    private function rewriteHtml(string $html, string $baseUrl): string {
        $parsed = parse_url($baseUrl);
        $baseDomain = ($parsed['scheme'] ?? 'https') . '://' . ($parsed['host'] ?? '');
        $basePath = isset($parsed['path']) ? rtrim(dirname($parsed['path']), '/') . '/' : '/';

        $html = preg_replace('/<html/i', '<!--nobanner--><html', $html, 1);
        $html = preg_replace_callback('/(href|src|action)=["\']([^"\']+)["\']/i', function($matches) use ($baseDomain, $basePath) {
            $url = $matches[2];
            if (preg_match('/^(#|javascript:|data:|mailto:)/i', $url)) return $matches[0];
            
            $absolute = $this->resolveUrl($url, $baseDomain, $basePath);
            
            // 【V3.5 核心修改】: URL编码非 ASCII 字符 (修复中文路径问题)
            $absolute = $this->encodeUrlPath($absolute);

            $newUrl = '?u=' . $this->urlsafe_base64_encode($absolute);
            return $matches[1] . '="' . $newUrl . '"';
        }, $html);

        // --- 【融合 V3.5 新增】: 注入前端脚本和 Service Worker 注册逻辑 ---
        $script_injection = "<script>{$this->JS_PROXY_REWRITER}</script>";
        $script_injection .= "<script>{$this->JS_SW_REGISTRATION}</script>";

        // 将脚本注入到 </body> 标签之前
        if (stripos($html, '</body>') !== false) {
            $html = str_ireplace('</body>', $script_injection . '</body>', $html);
        } else {
            // 如果没有 body 标签，注入到 HTML 文件的末尾
            $html .= $script_injection;
        }
        // -------------------------------------------------------------

        return $html;
    }

    private function rewriteCss(string $css, string $baseUrl): string {
        $parsed = parse_url($baseUrl);
        $baseDomain = ($parsed['scheme'] ?? 'https') . '://' . ($parsed['host'] ?? '');
        $basePath = isset($parsed['path']) ? rtrim(dirname($parsed['path']), '/') . '/' : '/';

        return preg_replace_callback('/url\s*\(\s*([\'"]?)([^\'"\)]+)\1\s*\)/i', function($matches) use ($baseDomain, $basePath) {
            $url = $matches[2];
            if (preg_match('/^(data:|https?:|\/\/)/i', $url)) return $matches[0];
            $absolute = $this->resolveUrl($url, $baseDomain, $basePath);
            
            $absolute = $this->encodeUrlPath($absolute); 

            return 'url(' . '?u=' . $this->urlsafe_base64_encode($absolute) . ')';
        }, $css);
    }

    private function resolveUrl(string $url, string $baseDomain, string $basePath): string {
        if (str_starts_with($url, '//')) return 'https:' . $url;
        if (str_starts_with($url, '/')) return $baseDomain . $url;
        if (preg_match('#^https?://#i', $url)) return $url;
        return $baseDomain . $basePath . $url;
    }

    // ========== 其他辅助功能 ==========

    private function streamDirectly(string $url, string $ua): void {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => $ua,
            CURLOPT_WRITEFUNCTION => function($ch, $chunk) { echo $chunk; return strlen($chunk); }
        ]);
        curl_exec($ch);
        curl_close($ch);
    }

    private function detectContentTypeFromBytes(string $content): ?string {
        if (strlen($content) < 4) return null;
        $firstFour = substr($content, 0, 4);
        
        if ($firstFour === "\x50\x4b\x03\x04") return 'application/zip';
        if (substr($firstFour, 0, 3) === 'GIF') return 'image/gif';
        if ($firstFour === "\x89PNG") return 'image/png';
        if ($firstFour === '%PDF') return 'application/pdf';
        
        return null;
    }

    private function securityCheck(string $url): bool { return true; }
    private function encryptAes(string $data): string {
        $iv = openssl_random_pseudo_bytes(16);
        $encrypted = openssl_encrypt($data, 'AES-256-CBC', $this->key, OPENSSL_RAW_DATA, $iv);
        return $this->urlsafe_base64_encode($iv . $encrypted);
    }
    private function decryptAes(string $data): string {
        $data = $this->urlsafe_base64_decode($data);
        if (strlen($data) < 17) return '';
        $iv = substr($data, 0, 16);
        $raw = substr($data, 16);
        return openssl_decrypt($raw, 'AES-256-CBC', $this->key, OPENSSL_RAW_DATA, $iv) ?: '';
    }
    private function urlsafe_base64_encode(string $data): string {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    private function urlsafe_base64_decode(string $data): string {
        return base64_decode(strtr($data, '-_', '+/') . str_repeat('=', (4 - strlen($data) % 4) % 4));
    }
    private function normalizeUrl(string $url): string {
        $url = trim($url);
        if (!preg_match('#^https?://#', $url)) $url = 'http://' . $url;
        return str_replace(' ', '%20', $url);
    }

    private function renderForm(): void {
        ?>
        <!--nobanner-->
        <!DOCTYPE html>
        <html lang="zh">
        <head>
            <meta charset="UTF-8">
            <title>🚀 极简代理</title>
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body { background: #1a1a1a; color: #eee; font-family: sans-serif; display: flex; height: 100vh; justify-content: center; align-items: center; margin: 0; }
                .box { background: #2d2d2d; padding: 2rem; border-radius: 8px; box-shadow: 0 4px 15px rgba(0,0,0,0.5); width: 100%; max-width: 500px; }
                h2 { color: #007bff; margin-bottom: 0.5rem; }
                input { width: 93%; padding: 12px; margin-bottom: 1rem; border: 1px solid #444; background: #222; color: #fff; border-radius: 4px; }
                button { width: 100%; padding: 12px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 1rem; }
                button:hover { background: #0056b3; }
                .tips { color: #888; font-size: 0.85rem; margin-top: 1rem; line-height: 1.5; }
                code { background: #000; padding: 2px 4px; border-radius: 3px; color: #0f0; }
                .badge { background: #e91e63; color: white; font-size: 0.7rem; padding: 2px 6px; border-radius: 4px; vertical-align: middle; }
            </style>
        </head>
        <body>
            <div class="box">
                <h2>🚀 极简代理 <sup><span class="badge">v1.0 beta</span></sup></h2>
                <form method="POST">
                    <input type="text" name="target_url" placeholder="输入目标 URL (如 https://raw.githubusercontent.com/...)" required>
                    <button type="submit">立即加速</button>
                </form>
                <div class="tips">
                    <p>支持功能：<br>简单页面浏览<br>AES 加密链接保护<br>自动重写页面资源链接</p>
                    <p>快捷方式：<code>/https://github.com/kooker</code><br><code>/https://raw.githubusercontent.com/cmliu/CF-Workers-GitHub/refs/heads/main/_worker.js</code></p>
                    <p>注：不支持搜索类、交互类、复杂JS类网站</p>
                </div>
            </div>
            <script><?php echo $this->JS_PROXY_REWRITER; ?></script>
            <script><?php echo $this->JS_SW_REGISTRATION; ?></script>
        </body>
        </html>
        <?php
    }
}

// 启动代理
(new ProxyHandler())->run();